/*************************************************/
/* Examples Program For "ET-dsPIC30F2010"  Board */
/* Hardware    : ET-dsPIC30F2010 TRAINING KIT V1 */
/* Target MCU  : dsPIC30F2010                    */
/*       	   : X-TAL : 7.3728 MHz              */
/*             : Run 117.9648MHz                 */
/*             : Selec OSC Mode = XT w/PLL 16x   */
/* Compiler    : MPLAB + C30 V1.33  		     */
/* Last Update : 1/October/2005                  */
/* Function    : Example Use ADC0                */
/*             : Read ADC0 & Display to UART1    */
/*************************************************/
/* Used RB0..RB3 = ADC Input                     */
/* Display Result to UART1 : 9600,N,8,1          */
/*************************************************/

#include "p30f2010a2.h" 	                           		// For dsPIC30F2010 MPU Register
#include "uart.h"											// Used UART Library Function
#include "adc10.h"											// Used ADC Library Function
#include "stdio.h"											// Used "sprintf" Function

/* Setup Configuration For ET-dsPIC30F2010 */
_FOSC(CSW_FSCM_ON & XT_PLL16);								// Enable Clock Switching,Enable Fail-Salf Clock
                                                            // Closk Source = Primary XT + (PLL x 16)
_FWDT(WDT_OFF);												// Disable Watchdog 
_FBORPOR(PBOR_ON & BORV_45 & PWRT_64 & MCLR_EN);			// Enable Brown-Out = 4.5V,Power ON = 64mS,Enable MCLR
_FGS(CODE_PROT_OFF);										// Code Protect OFF
/* End   Configuration For ET-dsPIC30F2010 */

unsigned char uart_buf[40];									// "sprintf" Buffer
unsigned int  adc_buff[4];									// 4 Channel ADC Result
unsigned int  adc_count;

/* pototype  section */
void init_uart(void);										// Initial UART1 Function
void init_adc(void);										// Initial ADC Function

int main(void)
{  
  init_uart();												// Initial UART = 9600,N,8,1
  init_adc();												// Initial ADC[0..3]
  	
  sprintf(uart_buf,"ET-dsPIC30F2010 TRAINING KIT V1\n\r"); 	// Print Message String 
  putsUART1((unsigned int *)uart_buf);                      // Print uart_buff to UART1
  while(BusyUART1());										// Wait putsUART1 Complete
  
  sprintf(uart_buf,"Demo ADC[0..3] Scan Read Input\n\n\r");	// Print Message String 
  putsUART1((unsigned int *)uart_buf);                      // Print uart_buff to UART1
  while(BusyUART1());										// Wait putsUART1 Complete  

  // Strat Read ADC Continue //
  while(1)													// Loop Continue
  {
    for(adc_count=0;adc_count<4;adc_count++)				// Loop Read ADC[0..3] 
    {
      ADCON1bits.SAMP = 1;				    				// Start Sampling ADC Now
      while(!ADCON1bits.SAMP);								// Wait ADC Sampling Complete
	  ConvertADC10();										// Convert Result
	  while(BusyADC10());									// Wait Conver Result Complete
      adc_buff[adc_count] = ReadADC10(adc_count);			// Save Result to Buffer    
    }

    // Display ADC[0..3] Result on UART1
    sprintf(uart_buf,"\rADC[0..3] = "); 					// Print Message String 
    putsUART1((unsigned int *)uart_buf);                    // Print uart_buff to UART1
    while(BusyUART1());										// Wait putsUART1 Complete  
    
    // Display Result ADC0    	
    sprintf(uart_buf,"%4d : ",adc_buff[0]);					// 4 Digit Decimal[0..1023] Display
    putsUART1((unsigned int *)uart_buf);                    // Print uart_buff to UART1
    while(BusyUART1());										// Wait putsUART1 Complete 
    
    // Display Result ADC1    	
    sprintf(uart_buf,"%4d : ",adc_buff[1]); 				// 4 Digit Decimal[0..1023] Display
    putsUART1((unsigned int *)uart_buf);                    // Print uart_buff to UART1
    while(BusyUART1());										// Wait putsUART1 Complete 

    // Display Result ADC2    
    sprintf(uart_buf,"%4d : ",adc_buff[2]); 				// 4 Digit Decimal[0..1023] Display
    putsUART1((unsigned int *)uart_buf);                    // Print uart_buff to UART1
    while(BusyUART1());										// Wait putsUART1 Complete 

    // Display Result ADC3    
	sprintf(uart_buf,"%4d \r",adc_buff[3]); 				// 4 Digit Decimal[0..1023] Display
    putsUART1((unsigned int *)uart_buf);                    // Print uart_buff to UART1
    while(BusyUART1());										// Wait putsUART1 Complete     
  } 
}	  

/*********************************/
/* Initial UART for dsPIC30F2010 */
/* 9600,N,8,1 / 117.9648MHz Fosc */
/*********************************/
void init_uart()
{		  
  CloseUART1();												// Disable UART1 Before New Config

  // Config UART1 Interrupt Control
  ConfigIntUART1(UART_RX_INT_DIS &							// Disable RX Interrupt
    		     UART_RX_INT_PR2 &							// RX Interrupt Priority = 2
    		     UART_TX_INT_DIS &							// Disable TX Interrupt
    		     UART_TX_INT_PR3);							// TX Interrupt Priority = 3  
  // Open UART1 = Mode,Status,Baudrate              
  OpenUART1(UART_EN	&										// Enable UART(UART Mode)
            UART_IDLE_STOP &								// Disable UART in IDLE Mode 
 			UART_ALTRX_ALTTX & 								// Select U1TX=RC13,U1RX=RC14
            UART_DIS_WAKE &									// Disable Wake-Up
			UART_DIS_LOOPBACK &								// Disable Loop Back
			UART_DIS_ABAUD &								// Disable Auto Baudrate
  			UART_NO_PAR_8BIT &								// UART = 8 Bit, No Parity
 			UART_1STOPBIT,									// UART = 1 Stop Bit

	  		// Config UART1 Status
  			UART_INT_TX & 									// Select Interrupt After TX Complete
	 		UART_TX_PIN_NORMAL &							// Normal U1TX Mode
 			UART_TX_ENABLE &								// Enable U1TX
 	 		UART_INT_RX_BUF_FUL &							// Flasg Set After RX Complete 
  			UART_ADR_DETECT_DIS &              				// Disable Check Address 
			UART_RX_OVERRUN_CLEAR,							// Clear Overrun Flag

  			// ET-dsPIC30F2010 TRAINING KIT V1.0 Hardware Board
  			// XTAL = 7.3728MHz
  			// Fosc = 7.3728 MHz x 16 = 117.9648 MHz
  			// Fcy(UART) = Fosc / 4 
  			//           = 117.9648 / 4 = 29.4912 MHz
  			// U1BRG = [Fcy/(16xBaud)]-1
  			//       = [29.4912 MHz / (16x9600)] - 1
  			//       = 191 = BFH			
  			191);											// ET-dsPIC30F2010 UART Baudrate = 9600 BPS
}

/********************************/
/* Initial ADC for dsPIC30F2010 */
/* Used RB[0..3] = 10 Bit ADC   */
/********************************/
void init_adc()
{
  ADCON1bits.ADON = 0;										// Turn-OFF ADC Before Change Config

  // Initial ADC Channel	
  SetChanADC10(ADC_CH0_POS_SAMPLEA_AN0 &					// ADC0 = Input(+)
			   ADC_CH0_POS_SAMPLEA_AN1 &					// ADC1 = Input(+)
			   ADC_CH0_POS_SAMPLEA_AN2 &					// ADC2 = Input(+)
			   ADC_CH0_POS_SAMPLEA_AN3 &					// ADC3 = Input(+)
			   ADC_CH0_NEG_SAMPLEA_NVREF);					// GND  = Input(-)             

  // Initial ADC Interrupt
  ConfigIntADC10(ADC_INT_DISABLE);							// Dsiable ADC Interrupt
  
  // Open ADC Function & Turn ON ADC
  OpenADC10(ADC_MODULE_ON &									// Turn-ON ADC Function
       		ADC_IDLE_STOP &									// Stop ADC in IDLE Mode
			ADC_FORMAT_INTG &								// Result Format = Unsigned Integer 
			ADC_CLK_MANUAL &								// ADC Clock = Manual
			ADC_AUTO_SAMPLING_ON &							// Enable ADC Sampling 
            ADC_SAMPLE_SIMULTANEOUS,						// Sample Style = Simultaneous			

            // ADC Config2
			ADC_VREF_AVDD_AVSS &							// VDD=Vref(H),VSS=Vref(L)
			ADC_SCAN_ON &									// Enable Scan        
            ADC_CONVERT_CH0 &								// Used ADC0 to Convert Result
            ADC_SAMPLES_PER_INT_16 &						// Number of Sample Between Interrupt(MAX)
            ADC_ALT_BUF_OFF & 								// Disable Alternate Buffer
			ADC_ALT_INPUT_OFF ,								// Disable Alternate Input	          

			// ADC Config3
			ADC_SAMPLE_TIME_1 &	    						// Samplink Time = Fast
			ADC_CONV_CLK_INTERNAL_RC &						// Used Internal RC Clock Sampling
			//ADC_CONV_CLK_SYSTEM &							// Used System Clock Sampling
			ADC_CONV_CLK_Tcy,								// Conversion Clock = Fast

			// ADC Config Port = Enable RB[0..3] = ADC[0..3]
			ENABLE_AN0_ANA &								// Enable RB0 = ADC0
			ENABLE_AN1_ANA &								// Enable RB1 = ADC1
			ENABLE_AN2_ANA &								// Enable RB2 = ADC2
            ENABLE_AN3_ANA,									// Enable RB3 = ADC3
            
			// ADC Config Scan = ON ADC[0..3] , OFF ADC[4..7]
			SKIP_SCAN_AN4 & 								// Disable Scan ADC4
		    SKIP_SCAN_AN5 &									// Disable Scan ADC5
			SKIP_SCAN_AN6 & 								// Disable Scan ADC6
			SKIP_SCAN_AN7);									// Disable Scan ADC7
}
